<?php
/**
 * PHPDO - Library of PHP functions for intracting database using PDO
 * File: PHPDO.php - PHP 5.6
 * Author: GIANCK LUIZ - Humolot
 * Version: 2.3.7
 * Last Updated Date: 11/03/2019
 * Copyright (c) 2016. All Rights Reserved.
 */

Class PHPDO {

    public $columns = "*";              
    public $openBrackets = "";          
    public $closedBrackets = "";        
    public $dbTransaction = false;      
    public $commitTransaction = false;  
    public $andOrOperator = "AND";      
    public $backtick = "`";             
    public $orderByCols;                
    public $limit;                      
    public $groupByCols;                
    public $havingCondition;            
    public $totalRows;                  
    public $lastInsertId;               
    public $error;                      
    public $displayError = false;       
    public $rowsChanged;                
    public $dbSQLitePath;               
    public $fetchType;                  
    private $whereCondition = "";
    private $dbObj = NULL;
    private $connectionStatus = 0;
    private $sql;
    private $dbType;
    private $dbHostName;
    private $dbName;
    private $dbUserName;
    private $dbPassword;
    private $dbRollBack;
    private $dbTansactionStatus;
    private $values = array();
    private $subSQL = "";
    private $joinString = "";

    /*     * ******************* File related variables - use this for various file operations ******************************** */
    public $fileOutputMode = "browser";     
    public $checkFileName = true;           
    public $checkFileNameCharacters = true; 
    public $replaceOlderFile = false;       
    public $uploadedFileName = "";          
    public $fileUploadPath = "upload/";  
    public $maxSize = 100000;               
    public $fileSavePath = "saved/";     
    public $pdfFontName = "helvetica";      
    public $pdfFontSize = "8";              
    public $pdfFontWeight = "B";            
    public $pdfAuthorName = "PHPDO";  
    public $pdfSubject = "PDF"; 
    public $excelFormat = "2007";          
    public $outputXML = "";                  
    public $rootElement = "phpdo";          
    public $encoding = "utf-8";            
    public $rowTagName = "";               
    public $append = false;               
    public $existingFilePath;              
    public $delimiter = ",";               
    public $enclosure = '"';               
    public $isFile = false;                
    public $useFirstRowAsTag = false;
    public $outputHTML = "";               
    public $tableCssClass = "tblCss";      
    public $trCssClass = "trCss";          
    public $htmlTableStyle = "";           
    public $htmlTRStyle = "";              
    public $htmlTDStyle = "";              

    /*     * ****************************************** PDO Functions ********************************************************* */

    public function __construct() {
        
    }

    public function conexao($hostName, $userName, $password, $databaseName, $databasePorta = "3306", $dbType = "mysql", $charset = "utf8") {
        $this->dbType = strtolower($dbType);
        $this->dbHostName = $hostName;
        $this->dbName = $databaseName;
		$this->dbPort = $databasePorta;
        $this->dbUserName = $userName;
        $this->dbPassword = $password;
        $this->characterSet = $charset;
        $this->dbObj = NULL;
        $this->connectionStatus = 0;

        if ($this->dbType === "sqlite" && !empty($databaseName))
            $this->dbSQLitePath = $databaseName;

        switch ($this->dbType) {
            case "mysql": $this->connectMysql();
                break;
            case "pgsql": $this->connectPGsql();
                break;
            case "sqlite": $this->connectSQLite();
                break;
            default: $this->setErrors("Por favor, indique uma opo de banco de dados vlido. As opes disponveis so MySql, PgSql e SQLite");
                return false;
        }

        if ($this->connectionStatus == 1) {
            $this->dbObj->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        } else {
            $this->setErrors("No foi possvel conectar ao banco de dados. Por favor, verifique os detalhes");
            return false;
        }

        if ($this->characterSet && $this->dbType !== "sqlite")
            $this->dbObj->exec("set names '" . $this->characterSet . "'");

        return $this;
    }

    private function connectMysql() {

        try {
			$this->dbObj = new PDO("mysql:host=$this->dbHostName;dbname=$this->dbName;port=$this->dbPort", $this->dbUserName, $this->dbPassword);
            $this->connectionStatus = 1;
        } catch (PDOException $e) {
            $this->setErrors($e->getMessage());
        }
    }

    private function connectPGsql() {

        try {
            $this->dbObj = new PDO("pgsql:dbname=$this->dbName;host=$this->dbHostName;user=$this->dbUserName;password=$this->dbPassword");
            $this->connectionStatus = 1;
            $this->backtick = "";
        } catch (PDOException $e) {
            $this->setErrors($e->getMessage());
        }
    }

    private function connectSQLite() {

        try {
            $this->dbObj = new PDO("sqlite:$this->dbSQLitePath");
            $this->connectionStatus = 1;
        } catch (PDOException $e) {
            $this->setErrors($e->getMessage());
        }
    }

    public function commitTransaction() {
        try {
            if ($this->dbObj != NULL) {
                $this->dbObj->commit();
                $this->beginTransaction = false;
            }
        } catch (PDOException $e) {
            $this->setErrors($e->getMessage());
        }
    }

    public function inserir($dbTableName, $insertData) {

        try {
            if ($this->dbTransaction && $this->dbRollBack == true)
                return;

            if ($this->dbTransaction && $this->dbTansactionStatus == 0) {
                $this->dbObj->beginTransaction();
                $this->dbTansactionStatus = 1;
            }

            $this->sql = $this->getInsertQuery($dbTableName, $insertData);
            $stmt = $this->dbObj->prepare($this->sql);
            $stmt->execute($this->values);
            $this->rowsChanged = $stmt->rowCount();
            $this->lastInsertId = $this->dbObj->lastInsertId();
            $this->resetAll();
            } catch (PDOException $e) {
            if ($this->dbTransaction == true) {
                $this->dbRollBack = true;
                $this->dbObj->rollBack();
            }

            $this->setErrors($e->getMessage());
        }
    }

    public function inserirBatch($dbTableName, $insertBatchData) {

        try {
            if ($this->dbTransaction && $this->dbRollBack == true)
                return;

            if ($this->dbTransaction && $this->dbTansactionStatus == 0) {
                $this->dbObj->beginTransaction();
                $this->dbTansactionStatus = 1;
            }

            foreach ($insertBatchData as $insertData) {
                $this->sql = $this->getInsertQuery($dbTableName, $insertData);
                $stmt = $this->dbObj->prepare($this->sql);
                $stmt->execute($this->values);
                $this->rowsChanged += $stmt->rowCount();
                $this->lastInsertId = $this->dbObj->lastInsertId();
                $this->resetAll();
            }
        } catch (PDOException $e) {
            if ($this->dbTransaction == true) {
                $this->dbRollBack = true;
                $this->dbObj->rollBack();
            }

            $this->setErrors($e->getMessage());
        }
    }

    public function atualizar($dbTableName, $updateData) {

        try {
            if ($this->dbTransaction && $this->dbRollBack == true)
                return;

            if ($this->dbTransaction && $this->dbTansactionStatus == 0) {
                $this->dbObj->beginTransaction();
                $this->dbTansactionStatus = 1;
            }
            $this->sql = $this->getUpdateQuery($dbTableName, $updateData);
            $stmt = $this->dbObj->prepare($this->sql);
            $stmt->execute($this->values);
            $this->rowsChanged = $stmt->rowCount();
            $this->resetAll();
        } catch (PDOException $e) {
            if ($this->dbTransaction == true) {
                $this->dbRollBack = true;
                $this->dbObj->rollBack();
            }
            $this->setErrors($e->getMessage());
        }
    }

    public function atualizarBatch($dbTableName, $updateBatchData, $where = array()) {

        try {
            if ($this->dbTransaction && $this->dbRollBack == true)
                return;

            if ($this->dbTransaction && $this->dbTansactionStatus == 0) {
                $this->dbObj->beginTransaction();
                $this->dbTansactionStatus = 1;
            }
            $loop = 0;
            foreach ($updateBatchData as $updateData) {
                if (isset($where[$loop])) {
                    $operator = isset($where[$loop][2]) ? $where[$loop][2] : "=";
                    $this->where($where[$loop][0], $where[$loop][1], $operator);
                }
                $this->sql = $this->getUpdateQuery($dbTableName, $updateData);
                $stmt = $this->dbObj->prepare($this->sql);
                $stmt->execute($this->values);
                $this->rowsChanged = $stmt->rowCount();
                $this->resetAll();
                $loop++;
            }
        } catch (PDOException $e) {
            if ($this->dbTransaction == true) {
                $this->dbRollBack = true;
                $this->dbObj->rollBack();
            }
            $this->setErrors($e->getMessage());
        }
    }

    public function excluir($dbTableName) {
        try {
            if ($this->dbTransaction && $this->dbRollBack)
                return;

            if ($this->dbTransaction && $this->dbTansactionStatus == 0) {
                $this->dbObj->beginTransaction();
                $this->dbTansactionStatus = 1;
            }
            $this->sql = $this->getDeleteQuery($dbTableName);
            $stmt = $this->dbObj->prepare($this->sql);
            $stmt->execute($this->values);
            $this->rowsChanged = $stmt->rowCount();
            $this->resetAll();
        } catch (PDOException $e) {
            if ($this->dbTransaction == true) {
                $this->dbRollBack = true;
                $this->dbObj->rollBack();
            }
            $this->setErrors($e->getMessage());
        }
    }

    public function excluirBatch($dbTableName, $where = array()) {
        try {
            if ($this->dbTransaction && $this->dbRollBack)
                return;

            if ($this->dbTransaction && $this->dbTansactionStatus == 0) {
                $this->dbObj->beginTransaction();
                $this->dbTansactionStatus = 1;
            }

            for ($loop = 0; $loop < count($where); $loop++) {
                if (isset($where[$loop])) {
                    $operator = isset($where[$loop][2]) ? $where[$loop][2] : "=";
                    $this->where($where[$loop][0], $where[$loop][1], $operator);
                }
                $this->sql = $this->getDeleteQuery($dbTableName);
                $stmt = $this->dbObj->prepare($this->sql);
                $stmt->execute($this->values);
                $this->rowsChanged = $stmt->rowCount();
                $this->resetAll();
            }
            } catch (PDOException $e) {
            if ($this->dbTransaction == true) {
                $this->dbRollBack = true;
                $this->dbObj->rollBack();
            }
            $this->setErrors($e->getMessage());
        }
    }

    public function selecionar($dbTableName) {

        try {
            $this->sql = $this->getSelectQuery($dbTableName);
            $stmt = $this->dbObj->prepare($this->sql);
            $stmt->execute($this->values);
            $result = $stmt->fetchAll($this->getFetchType());

            if (is_array($result))
                $this->totalRows = count($result);
            $this->resetAll();
            return $result;
            } catch (PDOException $e) {
            if ($this->dbTransaction == true) {
                $this->dbRollBack = true;
                $this->dbObj->rollBack();
            }
            $this->setErrors($e->getMessage());
        }
    }

    public function joinTables($joinTableName, $joinCondition, $joinType = "INNER JOIN") {
        $this->joinString .= $joinType . " " . $this->parseTable($joinTableName) . " ON " . $joinCondition . " ";
        return $this;
    }

    public function executeQuery($sql, $values = array()) {
        try {
            $this->sql = $sql;
            $stmt = $this->dbObj->prepare($this->sql);
            $this->values = $values;
            $stmt->execute($this->values);
            $result = $stmt->fetchAll($this->getFetchType());

            if (is_array($result))
                $this->totalRows = count($result);

            return $result;
        } catch (PDOException $e) {
            if ($this->dbTransaction == true) {
                $this->dbRollBack = true;
                $this->dbObj->rollBack();
            }
            $this->setErrors($e->getMessage());
        }
    }

    public function where($column, $value, $operator = "=") {

        if (empty($this->whereCondition))
            $this->whereCondition = " WHERE ";
        else
            $this->whereCondition .= $this->andOrOperator . " ";

        if (!empty($this->openBrackets)) {
            $this->whereCondition .= $this->openBrackets . " ";
            $this->openBrackets = "";
        }

        if (strtoupper($operator) == "NOT IN" || strtoupper($operator) == "IN") {
            if (is_array($value)) {
                $parameters = array_map(function($val) {
                    return "?";
                }, $value);
            }
            $this->whereCondition .= implode(" ", $this->parseColumns((array) $column)) . strtoupper($operator) . " (" . implode($parameters, ",") . ") ";
        } else if (strtoupper($operator) == "BETWEEN") {
            $this->whereCondition .= implode(" ", $this->parseColumns((array) $column)) . " BETWEEN ? AND ? ";
        } else {
            $this->whereCondition .= implode(" ", $this->parseColumns((array) $column)) . $operator . " ? ";
        }

        if (!empty($this->closedBrackets)) {
            $this->whereCondition .= $this->closedBrackets . " ";
            $this->closedBrackets = "";
        }

        if (is_array($value))
            $this->values = array_merge($this->values, $value);
        else
            $this->values = array_merge($this->values, array($value));

        return $this;
    }

    public function where_subquery($column, $subquery, $operator, $value = "") {

        if (empty($this->whereCondition))
            $this->whereCondition = " WHERE ";
        else
            $this->whereCondition .= $this->andOrOperator . " ";

        if (!empty($this->openBrackets)) {
            $this->whereCondition .= $this->openBrackets . " ";
            $this->openBrackets = "";
        }

        $this->whereCondition .= implode(" ", $this->parseColumns((array) $column)) . " " . strtoupper($operator) . " (" . $subquery . ") ";

        if (!empty($this->closedBrackets)) {
            $this->whereCondition .= $this->closedBrackets . " ";
            $this->closedBrackets = "";
        }

        if (is_array($value))
            $this->values = array_merge($this->values, $value);
        else if (!empty($value))
            $this->values = array_merge($this->values, array($value));

        return $this;
    }

    public function subQuery($sql, $alias = "", $data = array()) {
        $this->subSQL .= ",(" . $sql . ")";

        if (!empty($alias))
            $this->subSQL .= " AS " . $alias;

        if (is_array($data) && count($data) > 0)
            $this->values = array_merge($data, $this->values);

        return $this;
    }

    public function getLastQuery() {
        return $this->sql;
    }

    public function setLimit($limit) {
        $this->limit = $limit;
        return $this;
    }

    public function setColumns($column = array()) {
        $this->columns = $column;
        return $this;
    }

    public function orderBy($colunm = array()) {
        $this->orderByCols = $colunm;
        return $this;
    }

    public function displayError($showError = true) {
        $this->displayError = $showError;
        return $this;
    }

    public function setBacktiks($backtick = "`") {
        $this->backtick = $backtick;
        return $this;
    }

    public function columnNames($dbTableName) {
        try {
            if ($this->dbType === "pgsql")
                $this->sql = "select column_name from INFORMATION_SCHEMA.COLUMNS where table_name = '" . $dbTableName . "'";
            else if ($this->dbType === "sqlite")
                $this->sql = "PRAGMA table_info('" . $dbTableName . "')";
            else
                $this->sql = "DESCRIBE " . $this->parseTable($dbTableName);
            $stmt = $this->dbObj->prepare($this->sql);
            $stmt->execute();
            $result = $stmt->fetchAll(PDO::FETCH_COLUMN);
            return $result;
        } catch (PDOException $e) {
            $this->setErrors($e->getMessage());
        }
    }

    public function getAllTables() {
        try {
            if ($this->dbType === "sqlite")
                $this->sql = "SELECT name FROM sqlite_master WHERE type='table'";
            else
                $this->sql = "SHOW TABLES FROM " . $this->dbName;
            $stmt = $this->dbObj->prepare($this->sql);
            $stmt->execute();
            $result = $stmt->fetchAll($this->getFetchType());
            return $result;
        } catch (PDOException $e) {
            $this->setErrors($e->getMessage());
        }
    }

    public function tableFieldInfo($dbTableName) {
        try {
            if ($this->dbType === "pgsql")
                $this->sql = "select column_name, data_type, character_maximum_length from INFORMATION_SCHEMA.COLUMNS where table_name = '" . $dbTableName . "'";
            else
                $this->sql = "SHOW FIELDS FROM " . $this->parseTable($dbTableName);
            $stmt = $this->dbObj->prepare($this->sql);
            $stmt->execute();
            $result = $stmt->fetchAll($this->getFetchType());
            return $result;
        } catch (PDOException $e) {
            $this->setErrors($e->getMessage());
        }
    }

    public function primaryKey($dbTableName) {
        try {
            if ($this->dbType === "pgsql")
                $this->sql = "SELECT a.attname, format_type(a.atttypid, a.atttypmod) AS data_type FROM   pg_index i JOIN   pg_attribute a ON a.attrelid = i.indrelid AND a.attnum = ANY(i.indkey) WHERE  i.indrelid = '$dbTableName'::regclass AND    i.indisprimary;";
            else
                $this->sql = "SHOW INDEXES FROM " . $this->parseTable($dbTableName) . " WHERE Key_name = 'PRIMARY'";
            $stmt = $this->dbObj->prepare($this->sql);
            $stmt->execute();
            $result = $stmt->fetchAll($this->getFetchType());
            if (count($result) > 0) {
                if ($this->dbType === "pgsql")
                    return $result[0]["attname"];
                else
                    return $result[0]["Column_name"];
            } else
                return "";
        } catch (PDOException $e) {
            $this->setErrors($e->getMessage());
        }
    }

    public function dropTable($dbTableName) {
        try {
            $this->sql = "DROP TABLE " . $this->parseTable($dbTableName);
            $stmt = $this->dbObj->prepare($this->sql);
            $stmt->execute();
            return true;
        } catch (PDOException $e) {
            $this->setErrors($e->getMessage());
        }
    }

    public function truncateTable($dbTableName) {
        try {
            if ($this->dbType === "sqlite")
                $this->sql = "DELETE FROM " . $this->parseTable($dbTableName);
            else
                $this->sql = "TRUNCATE TABLE " . $this->parseTable($dbTableName);
            $stmt = $this->dbObj->prepare($this->sql);
            $stmt->execute();
            return true;
        } catch (PDOException $e) {
            $this->setErrors($e->getMessage());
        }
    }

    public function renameTable($oldDBTableName, $newDBTableName) {
        try {
            if ($this->dbType === "mysql")
                $this->sql = "RENAME  TABLE " . $this->parseTable($oldDBTableName) . " TO " . $this->parseTable($newDBTableName);
            else
                $this->sql = "ALTER TABLE " . $this->parseTable($oldDBTableName) . " RENAME TO " . $this->parseTable($newDBTableName);

            $stmt = $this->dbObj->prepare($this->sql);
            $stmt->execute();
            return true;
        } catch (PDOException $e) {
            $this->setErrors($e->getMessage());
        }
    }

    private function getInsertQuery($dbTableName, $insertData) {
        $this->columns = implode(",", $this->parseColumns(array_keys($insertData)));
        $this->values = array_values($insertData);
        $this->parameters = "";

        $this->parameters = array_map(function($val) {
            return "?";
        }, $insertData);


        $this->parameters = implode($this->parameters, ",");
        return "INSERT INTO " . $this->parseTable($dbTableName) . " ($this->columns) VALUES ($this->parameters)";
    }

    private function getUpdateQuery($dbTableName, $updateData) {
        $this->columns = implode("=?,", $this->parseColumns(array_keys($updateData))) . "=?";
        $this->values = array_merge(array_values($updateData), $this->values);
        $whereCondition = $this->whereCondition;
        $orderByCondition = $this->getorderByCondition();
        $limit = $this->getLimitCondition();

        return "UPDATE " . $this->parseTable($dbTableName) . " SET $this->columns $whereCondition $orderByCondition $limit";
    }

    private function getDeleteQuery($dbTableName) {
        $whereCondition = $this->whereCondition;
        $orderByCondition = $this->getorderByCondition();
        $limit = $this->getLimitCondition();

        return "DELETE FROM " . $this->parseTable($dbTableName) . " $whereCondition $orderByCondition $limit";
    }

    private function getSelectQuery($dbTableName) {

        if (is_array($this->columns) && count($this->columns) > 0)
            $cols = implode(",", array_values($this->parseColumns($this->columns)));
        else
            $cols = "* ";

        if (!empty($this->closedBrackets)) {
            $this->whereCondition .= $this->closedBrackets . " ";
            $this->closedBrackets = "";
        }

        $sql = "SELECT " . $cols . $this->subSQL . " FROM " . $this->parseTable($dbTableName) . " " . $this->joinString . " " . $this->whereCondition
                . $this->getGroupByCondition() . $this->getHavingCondition() . $this->getorderByCondition() . $this->getLimitCondition();

        return $sql;
    }

    /* Returns order by  condition */

    private function getorderByCondition() {
        $orderBy = "";
        if (is_array($this->orderByCols) && count($this->orderByCols) > 0)
            $orderBy.=" ORDER BY " . implode(",", $this->parseColumns($this->orderByCols));

        return $orderBy;
    }

    private function getLimitCondition() {
        $limitBy = "";
        if (!empty($this->limit)) {
            if ($this->dbType === "pgsql") {
                $limit = explode(",", $this->limit);
                $limitBy.=" LIMIT " . $limit[0];
                if (isset($limit[1]))
                    $limitBy.=" OFFSET " . $limit[1];
            } else {
                $limitBy.=" LIMIT " . $this->limit;
            }
        }

        return $limitBy;
    }

    private function getGroupByCondition() {
        $groupby = "";
        if (!empty($this->groupByCols))
            $groupby = " GROUP BY " . implode(",", $this->parseColumns($this->groupByCols));

        return $groupby;
    }

    private function getHavingCondition() {
        $having = "";
        if ($this->havingCondition)
            $having.=" HAVING " . implode(",", $this->havingCondition);

        return $having;
    }

    private function getFetchType() {
        switch (strtoupper($this->fetchType)) {
            case "BOTH": return PDO::FETCH_BOTH;
            case "NUM": return PDO::FETCH_NUM;
            case "ASSOC": return PDO::FETCH_ASSOC;
            case "OBJ": return PDO::FETCH_OBJ;
            case "COLUMN":return PDO::FETCH_COLUMN;
            default: return PDO::FETCH_ASSOC;
        }
    }

    public function resetAll() {
        $this->whereCondition = "";
        $this->joinString = "";
        $this->values = array();
        $this->limit = "";
        $this->columns = array();
        $this->groupByCols = array();
        $this->havingCondition = array();
        $this->orderByCols = array();
        $this->subSQL = "";
    }

    public function resetWhere() {
        $this->whereCondition = "";
    }

    public function resetValues() {
        $this->values = array();
    }

    public function resetLimit() {
        $this->limit = "";
    }

    public function resetError() {
        $this->error = array();
    }

    private function parseColumns($cols) {
        $columns = array();
        if (is_array($cols) && !empty($this->backtick)) {
            foreach ($cols as $col) {
                if ($this->checkColAggr($col)) {
                    $alias = explode(" as ", strtolower($col));
                    if (isset($alias[1]))
                        $col = trim($alias[0]) . " AS " . $this->backtick . trim($alias[1]) . $this->backtick;
                }
                else if (preg_match("/\bas\b/i", strtolower($col))) {
                    list($colName, $table) = explode(" as ", strtolower($col));
                    if (strpos($colName, ".") !== false) {
                        $colData = explode(".", $colName);
                        $col = $this->backtick . trim($colData[0]) . $this->backtick . "." . $this->backtick . trim($colData[1]) . $this->backtick . " AS " . $this->backtick . trim($table) . $this->backtick;
                    }
                } else if (preg_match("/\basc\b/i", strtolower($col))) {
                    list($colName) = explode(" asc", strtolower($col));
                    if (strpos($colName, ".") !== false) {
                        $colData = explode(".", $colName);
                        $col = $this->backtick . trim($colData[0]) . $this->backtick . "." . $this->backtick . trim($colData[1]) . $this->backtick . " ASC";
                    } else {
                        $col = $this->backtick . trim($colName) . $this->backtick . " ASC";
                    }
                } else if (preg_match("/\bdesc\b/i", strtolower($col))) {
                    list($colName) = explode(" desc", strtolower($col));
                    if (strpos($colName, ".") !== false) {
                        $colData = explode(".", $colName);
                        $col = $this->backtick . trim($colData[0]) . $this->backtick . "." . $this->backtick . trim($colData[1]) . $this->backtick . " DESC";
                    } else {
                        $col = $this->backtick . trim($colName) . $this->backtick . " DESC";
                    }
                } else {
                    if (strpos($col, ".") !== false) {
                        $val = explode(".", $col);
                        $col = $this->backtick . trim($val[0]) . $this->backtick . "." . $this->backtick . trim($val[1]) . $this->backtick;
                    } else {
                        $col = $this->backtick . $col . $this->backtick;
                    }
                }

                $columns[] = $col;
            }
        } else {
            $columns = $cols;
        }
        return $columns;
    }

    private function parseTable($dbTable) {
        $table = $dbTable;
        if (!empty($this->backtick)) {
            if (preg_match("/\bas\b/i", strtolower($dbTable))) {
                list($tableName, $alias) = explode("as", strtolower($dbTable));
                $table = $this->backtick . trim($tableName) . $this->backtick . " AS " . $this->backtick . trim($alias) . $this->backtick;
            } else {
                $table = $this->backtick . trim($dbTable) . $this->backtick;
            }
        }
        return $table;
    }

    private function checkColAggr($column) {
        $col = strtolower($column);
        if (strpos($col, "concat") !== false || strpos($col, "sum") !== false || strpos($col, "max") !== false || strpos($col, "min") !== false || strpos($col, "count") !== false || strpos($col, "distinct") !== false) {
            return true;
        }
        return false;
    }

    private function setErrors($error) {
        $this->error[] = $error;
        if ($this->displayError)
            echo $error;
    }

    public function arrayToCSV($csvArray, $fileName = "file.csv") {
        if (!is_array($csvArray)) {
            $this->setErrors("Please provide valid input. ");
            return false;
        }
        if (!$fileName) {
            $this->setErrors("Please provide the csv file name");
            return false;
        }
        if ($this->append && !isset($this->existingFilePath)) {
            $this->setErrors("Please provide existing file path, you want to append data ");
            return false;
        }
        $list = $csvArray;
        if ($this->fileSavePath && !is_dir($this->fileSavePath))
            mkdir($this->fileSavePath);

        if ($this->append)
            $fp = fopen($this->existingFilePath, 'a+');
        else
            $fp = fopen($this->fileSavePath . $fileName, 'w');

        foreach ($list as $fields) {
            fputcsv($fp, $fields, $this->delimiter, $this->enclosure);
        }

        if ($this->fileOutputMode == 'browser') {
            header("Content-type: application/csv");
            header("Content-Disposition: attachment; filename=" . $fileName);
            header("Pragma: no-cache");
            readfile($this->fileSavePath . $fileName);
            die(); //to prevent page output
        }

        fclose($fp);
        return true;
    }

    public function arrayToXML($xmlArray, $outputFileName = "file.xml") {
        if (!is_array($xmlArray)) {
            $this->setErrors("Please provide valid input.");
            return false;
        }
        $xmlObject = new SimpleXMLElement("<?xml version=\"1.0\" encoding=\"$this->encoding\" ?><$this->rootElement></$this->rootElement>");
        $this->generateXML($xmlArray, $xmlObject, $this->rootElement);
        if ($this->fileOutputMode == "browser")
            echo $xmlObject->asXML();
        else {
            if ($this->fileSavePath && !is_dir($this->fileSavePath))
                mkdir($this->fileSavePath);
            $xmlObject->asXML($this->fileSavePath . $outputFileName);
        }
        return true;
    }

    

    public function arrayToPDF($pdfArray, $outputFileName = "file.pdf") {
        if (!is_array($pdfArray)) {
            $this->setErrors("Please provide valid input. ");
            return false;
        }
        require_once(dirname(__FILE__) . "/library/tcpdf/tcpdf.php");
        $pdf = new TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
        $pdf->SetCreator(PDF_CREATOR);
        $pdf->SetFont($this->pdfFontName, $this->pdfFontWeight, $this->pdfFontSize, '', 'false');
        $pdf->SetAuthor($this->pdfAuthorName);
        $pdf->SetSubject($this->pdfSubject);
        $pdf->SetDefaultMonospacedFont(PDF_FONT_MONOSPACED);
        $pdf->setImageScale(PDF_IMAGE_SCALE_RATIO);
        $pdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM);
        if (@file_exists(dirname(__FILE__) . '/lang/eng.php')) {
            require_once(dirname(__FILE__) . '/lang/eng.php');
            $pdf->setLanguageArray($l);
        }
        $pdf->AddPage();
        $this->arrayToHTML($pdfArray, "file.html", true);
        $pdf->writeHTML($this->outputHTML, true, false, true, false, '');
        if ($this->fileOutputMode == "browser")
            $pdf->Output($outputFileName, 'I');
        else {
            if ($this->fileSavePath && !is_dir($this->fileSavePath))
                mkdir($this->fileSavePath);
            $pdf->Output($this->fileSavePath . $outputFileName, 'F');
        }
        return true;
    }

    public function arrayToExcel($excelArray, $outputFileName = "file.xlsx") {
        if (!is_array($excelArray)) {
            $this->setErrors("Please provide valid input.");
            return false;
        }
        if ($this->append && !isset($this->existingFilePath)) {
            $this->setErrors("Please provide existing file path, you want to append data");
            return false;
        }
        if (empty($outputFileName)) {
            if ($this->excelFormat == "2007")
                $outputFileName = "file.xlsx";
            else
                $outputFileName = "file.xls";
        }
        require_once(dirname(__FILE__) . "/library/PHPExcel/PHPExcel.php");

        if ($this->append) {
            require_once (dirname(__FILE__) . "/library/PHPExcel/PHPExcel/IOFactory.php");
            if (!file_exists($this->existingFilePath)) {
                $this->setErrors("Could not open " . $this->existingFilePath . " for reading! File does not exist.");
                return false;
            }
            $objPHPExcel = PHPExcel_IOFactory::load($this->existingFilePath);
        } else {
            $objPHPExcel = new PHPExcel();
        }
        $objPHPExcel->setActiveSheetIndex(0);

        $cells = array("A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z");
        $colCount = 1;

        if ($this->append)
            $colCount = $objPHPExcel->getActiveSheet()->getHighestRow() + 1;

        foreach ($excelArray as $rows) {
            $cellLoop = 0;
            foreach ($rows as $row) {
                $objPHPExcel->getActiveSheet()->setCellValue($cells[$cellLoop] . $colCount, $row);
                $cellLoop++;
            }
            $colCount++;
        }
        if ($this->excelFormat == "2007") {
            $objWriter = PHPExcel_IOFactory::createWriter($objPHPExcel, 'Excel2007');
        } else {
            $objWriter = PHPExcel_IOFactory::createWriter($objPHPExcel, 'Excel5');
        }
        if ($this->append) {
            $objWriter->save($this->existingFilePath);
        } else {
            if ($this->fileOutputMode == "browser") {
                if ($this->excelFormat == "2007")
                    header('Content-type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
                else
                    header('Content-type: application/vnd.ms-excel');

                header('Content-Disposition: attachment; filename="' . $outputFileName . '"');
                $objWriter->save('php://output');
                die();
            }
            else {
                if ($this->fileSavePath && !is_dir($this->fileSavePath))
                    mkdir($this->fileSavePath);
                $objWriter->save($this->fileSavePath . $outputFileName);
            }
        }

        return true;
    }

    private function generateXML($xmlArray, &$xmlObject, $rootElement = "root") {
        foreach ($xmlArray as $key => $value) {
            if (is_array($value)) {
                if (!is_numeric($key)) {
                    $subnode = $xmlObject->addChild("$key");
                    $this->generateXML($value, $subnode, $rootElement);
                } else {
                    $this->generateXML($value, $xmlObject, $rootElement);
                }
            } else {
                if (is_numeric($key)) {
                    $key = $rootElement;
                }
                $xmlObject->addChild("$key", "$value");
            }
        }
    }


    public function arrayToJson($jsonArray) {
        return var_dump(json_encode($jsonArray, true)); 
    }


    public function csvToArray($fileName) {
        if (empty($fileName)) {
            $this->setErrors("Please provide the csv file name");
            return false;
        }
        $csvArray = array();
        if (($handle = fopen($fileName, "r")) !== FALSE) {
            $rowIndex = 0;
            while (($lineArray = fgetcsv($handle, 0, $this->delimiter)) !== FALSE) {
                for ($colIndex = 0; $colIndex < count($lineArray); $colIndex++) {
                    $csvArray[$rowIndex][$colIndex] = $lineArray[$colIndex];
                }
                $rowIndex++;
            }
            fclose($handle);
        }
        $csvArray = $this->formatOutputArray($csvArray);
        return $csvArray;
    }


    public function excelToArray($fileName) {
        if (!$fileName) {
            $this->setErrors("Please provide the excel file name");
            return false;
        }
        require_once(dirname(__FILE__) . "/library/PHPExcel/PHPExcel/IOFactory.php");
        $objPHPExcel = PHPExcel_IOFactory::load($fileName);
        $excelArray = $objPHPExcel->getActiveSheet()->toArray(null, true, true, false);
        $excelArray = $this->formatOutputArray($excelArray);
        return $excelArray;
    }


    public function xmlToArray($xmlSource) {
        if ($this->isFile)
            $xml = file_get_contents($xmlSource);
        else
            $xml = $xmlSource;

        $xmlObject = new SimpleXMLElement($xml);
        $decodeArray = @json_decode(@json_encode($xmlObject), 1);
        foreach ($decodeArray as $newDecodeArray) {
            $returnArray = $newDecodeArray;
        }
        return $returnArray;
    }

    private function formatOutputArray($data) {
        $output = array();
        $loop = 0;
        if (isset($data) && count($data) > 0) {
            $columns = $data[0];
            foreach ($data as $row) {
                if ($loop > 0)
                    $output[] = array_combine($columns, $row);
                $loop++;
            }
        }
        return $output;
    }
    
    
    public function getRandomC(){
                  $nr =rand(48,120);
          return chr($nr);
    }
  
    public function phpdoencode($s,$cheie){
    $codObstrufucat = '';
    
    for($i=0; $i< strlen($s);$i++){  
            
                $asci =ord(substr($s,$i,1)) + $cheie; 
                $codObstrufucat .= chr($asci);
              
              if($i%$cheie == 0) {
                  $codObstrufucat .= $this->getRandomC();
              
              }
          }
          
          return $codObstrufucat;
  	  }

    public function phpdodecode($codObstrufucat,$cheie,$useUtf9Decoding= true){
          $decodObstrufucat ='';
       if($useUtf9Decoding) $codObstrufucat = utf8_decode($codObstrufucat);
              for($i=0; $i< strlen($codObstrufucat);$i++){
                   if($i%($cheie+1) != 1) {
                  
                          
                          $asci =ord(substr($codObstrufucat,$i,1)) - $cheie; 
                          
                          $decodObstrufucat .= chr($asci);
                  }
              }
          return $decodObstrufucat;
        }
    
    
    public function upload($arquivo, $pasta, $tipos, $nome = null){
    if(isset($arquivo)){
        $infos = explode(".", $arquivo["name"]);
 
        if(!$nome){
            for($i = 0; $i < count($infos) - 1; $i++){
                $nomeOriginal = $nomeOriginal . $infos[$i] . ".";
            }
        }
        else{
            $nomeOriginal = $nome . ".";
        }
 
        $tipoArquivo = $infos[count($infos) - 1];
 
        $tipoPermitido = false;
        foreach($tipos as $tipo){
            if(strtolower($tipoArquivo) == strtolower($tipo)){
                $tipoPermitido = true;
            }
        }
        if(!$tipoPermitido){
            $retorno["erro"] = "Tipo no permitido";
        }
        else{
            if(move_uploaded_file($arquivo['tmp_name'], $pasta . $nomeOriginal . $tipoArquivo)){
                $retorno["caminho"] = $pasta . $nomeOriginal . $tipoArquivo;
            }
            else{
                $retorno["erro"] = "Erro ao fazer upload";
            }
        }
    }
    else{
        $retorno["erro"] = "Arquivo nao setado";
    }
    return $retorno;
    }


    public function phpdopassword($length = 8, $allow_special_character = false) {
        $alphabet = "abcdefghijklmnopqrstuwxyzABCDEFGHIJKLMNOPQRSTUWXYZ0123456789";
        if ($allow_special_character)
            $alphabet.="!@#$%&*(){}[]<>,.";
        return substr(str_shuffle($alphabet), 0, $length);
    }

    public function paginacao($page = 1, $totalrecords, $limit = 10, $adjacents = 1, $url = "javascript:;") {
        $pagination = "";
        if ($totalrecords > 0) {
            if (!$limit)
                $limit = 15;
            if (!$page)
                $page = 1;

            $prev = $page - 1;
            $next = $page + 1;
            $lastpage = ceil($totalrecords / $limit);
            $lpm1 = $lastpage - 1;

            if ($lastpage > 1) {
                $pagination .= "<div class=\"pagination\"><ul>";

                //previous button
                if ($page > 1)
                    $pagination .= "<li><a class='phpdo-page' 'href=\"$url\" data-page=" . $prev . ">Anterior</a></li>";
                else
                    $pagination .= "<li class=\"disabled\"><a class='phpdo-page' data-page=" . $prev . " 'href=\"$url\">Anterior</a></li>";

                //pages	
                if ($lastpage < 7 + ($adjacents * 2)) { //not enough pages to bother breaking it up
                    for ($counter = 1; $counter <= $lastpage; $counter++) {
                        if ($counter == $page)
                            $pagination .= "<li class=\"active\"><a class='phpdo-page' 'href=\"$url\" data-page=" . $counter . ">$counter</a></li>";
                        else
                            $pagination .= "<li><a class='phpdo-page' 'href=\"$url\"  data-page=" . $counter . ">" . $counter . "</a></li>";
                    }
                }
                elseif ($lastpage >= 7 + ($adjacents * 2)) { //enough pages to hide some
                    //close to beginning; only hide later pages
                    if ($page < 1 + ($adjacents * 3)) {
                        for ($counter = 1; $counter < 4 + ($adjacents * 2); $counter++) {
                            if ($counter == $page)
                                $pagination .= "<li class=\"active\"><a class='phpdo-page' data-page=" . $counter . "  href=\"$url\">$counter</a></li>";
                            else
                                $pagination .= "<li><a class='phpdo-page' 'href=\"$url\"  data-page=" . $counter . ">" . $counter . "</a></li>";
                        }
                        $pagination .= "<li class=\"elipses\">...</li>";
                        $pagination .= "<li><a class='phpdo-page' 'href=\"$url\"  data-page=" . $lpm1 . ">" . $lpm1 . "</a></li>";
                        $pagination .= "<li><a class='phpdo-page' 'href=\"$url\"  data-page=" . $lastpage . ">" . $lastpage . "</a></li>";
                    }
                    //in middle; hide some front and some back
                    elseif ($lastpage - ($adjacents * 2) > $page && $page > ($adjacents * 2)) {
                        $pagination .= "<li><a class='phpdo-page' 'href=\"$url\"  data-page=\"1\">1</a></li>";
                        $pagination .= "<li><a class='phpdo-page' 'href=\"$url\"  data-page=\"2\">2</a></li>";
                        $pagination .= "<li class=\"elipses\">...</li>";
                        for ($counter = $page - $adjacents; $counter <= $page + $adjacents; $counter++) {
                            if ($counter == $page)
                                $pagination .= "<li class=\"active\"><a class='phpdo-page' 'href=\"$url\">$counter</a></li>";
                            else
                                $pagination .= "<li><a class='phpdo-page' 'href=\"$url\"  data-page=" . $counter . ">" . $counter . "</a></li>";
                        }
                        $pagination .= "<li class=\"elipses\">...</li>";
                        $pagination .= "<li><a class='phpdo-page' 'href=\"$url\"  data-page=" . $lpm1 . ">" . $lpm1 . "</a></li>";
                        $pagination .= "<li><a class='phpdo-page' 'href=\"$url\"  data-page=" . $lastpage . ">" . $lastpage . "</a></li>";
                    }
                    //close to end; only hide early pages
                    else {
                        $pagination .= "<li><a class='phpdo-page' 'href=\"$url\"  data-page=\"1\">1</a></li>";
                        $pagination .= "<li><a class='phpdo-page' 'href=\"$url\"  data-page=\"2\">2</a></li>";
                        $pagination .= "<li class=\"elipses\">...</li>";
                        for ($counter = $lastpage - (1 + ($adjacents * 3)); $counter <= $lastpage; $counter++) {
                            if ($counter == $page)
                                $pagination .= "<li class=\"active\"><a class='pdomodel-page' 'href=\"$url\">$counter</a></li>";
                            else
                                $pagination .= "<li><a class='phpdo-page' 'href=\"$url\"  data-page=" . $counter . ">" . $counter . "</a></li>";
                        }
                    }
                }

               
                if ($page < $counter - 1)
                    $pagination .= "<li><a class='phpdo-page' ' href=\"$url\"  data-page=" . $next . ">Prximo</a></li>";
                else
                    $pagination .= "<li class=\"disabled\">Prximo</li>";
                $pagination .= "</ul></div>\n";
            }
        }

        return $pagination;
    }

}


// Template Class PHPDO

class PHPDOTemplate {

		protected $vars = array();
		protected $values = array();
		private $properties = array();
		protected $instances = array();
		protected $modifiers = array();
		private $blocks = array();
		private $parents = array();
		private $parsed = array();
		private $finally = array();
		private $accurate;
		private static $REG_NAME = "([[:alnum:]]|_)+";

		public function __construct($filename, $accurate = false){
			$this->accurate = $accurate;
			$this->loadfile(".", $filename);
		}

		public function addFile($varname, $filename){
			if(!$this->exists($varname)) throw new \InvalidArgumentException("addFile: var $varname does not exist");
			$this->loadfile($varname, $filename);
		}

		public function __set($varname, $value){
			#if(!$this->exists($varname)) throw new \RuntimeException("var $varname does not exist");
			$stringValue = $value;
			if(is_object($value)){
				$this->instances[$varname] = $value;
				if(!isset($this->properties[$varname])) $this->properties[$varname] = array();
				if(method_exists($value, "__toString")) $stringValue = $value->__toString();
				else $stringValue = "Object";
			}
			$this->setValue($varname, $stringValue);
			return $value;
		}

		public function __get($varname){
			if(isset($this->values["{".$varname."}"])) return $this->values["{".$varname."}"];
			elseif(isset($this->instances[$varname])) return $this->instances[$varname];
			throw new \RuntimeException("var $varname does not exist");
		}

		public function exists($varname){
			return in_array($varname, $this->vars);
		}

		private function loadfile($varname, $filename) {
			if (!file_exists($filename)) throw new InvalidArgumentException("file $filename does not exist");
			// If it's PHP file, parse it
			if($this->isPHP($filename)){
				ob_start();
				require $filename;
				$str = ob_get_contents();
				ob_end_clean();
				$this->setValue($varname, $str);
			} else {
				// Reading file and hiding comments
				$str = preg_replace("/<!---.*?--->/smi", "", file_get_contents($filename));
			#	if (empty($str)) throw new InvalidArgumentException("file $filename is empty");
				$this->setValue($varname, $str);
				$blocks = $this->identify($str, $varname);
				$this->createBlocks($blocks);
			}
		}

		private function isPHP($filename){
			foreach(array('.php', '.php5', '.cgi') as $php){
				if(0 == strcasecmp($php, substr($filename, strripos($filename, $php)))) return true;
			}
			return false;
		}

		private function identify(&$content, $varname){
			$blocks = array();
			$queued_blocks = array();
			$this->identifyVars($content);
			foreach (explode("\n", $content) as $line) {
				if (strpos($line, "<!--")!==false) $this->identifyBlocks($line, $varname, $queued_blocks, $blocks);
			}
			return $blocks;
		}

		private function identifyBlocks(&$line, $varname, &$queued_blocks, &$blocks){
			$reg = "/<!--\s*BEGIN\s+(".self::$REG_NAME.")\s*-->/sm";
			preg_match($reg, $line, $m);
			if (1==preg_match($reg, $line, $m)){
				if (0==sizeof($queued_blocks)) $parent = $varname;
				else $parent = end($queued_blocks);
				if (!isset($blocks[$parent])){
					$blocks[$parent] = array();
				}
				$blocks[$parent][] = $m[1];
				$queued_blocks[] = $m[1];
			}
			$reg = "/<!--\s*END\s+(".self::$REG_NAME.")\s*-->/sm";
			if (1==preg_match($reg, $line)) array_pop($queued_blocks);
		}

		private function identifyVars(&$content){
			$r = preg_match_all("/{(".self::$REG_NAME.")((\-\>(".self::$REG_NAME."))*)?((\|.*?)*)?}/", $content, $m);
			if ($r){
				for($i=0; $i<$r; $i++){
					// Object var detected
					if($m[3][$i] && (!isset($this->properties[$m[1][$i]]) || !in_array($m[3][$i], $this->properties[$m[1][$i]]))){
						$this->properties[$m[1][$i]][] = $m[3][$i];
					}
					// Modifiers detected
					if($m[7][$i] && (!isset($this->modifiers[$m[1][$i]]) || !in_array($m[7][$i], $this->modifiers[$m[1][$i].$m[3][$i]]))){
						$this->modifiers[$m[1][$i].$m[3][$i]][] = $m[1][$i].$m[3][$i].$m[7][$i];
					}
					// Common variables
					if(!in_array($m[1][$i], $this->vars)){
						$this->vars[] = $m[1][$i];
					}
				}
			}
		}

		private function createBlocks(&$blocks) {
			$this->parents = array_merge($this->parents, $blocks);
			foreach($blocks as $parent => $block){
				foreach($block as $chield){
					if(in_array($chield, $this->blocks)) throw new \UnexpectedValueException("duplicated block: $chield");
					$this->blocks[] = $chield;
					$this->setBlock($parent, $chield);
				}
			}
		}

		private function setBlock($parent, $block) {
			$name = $block.'_value';
			$str = $this->getVar($parent);
			if($this->accurate){
				$str = str_replace("\r\n", "\n", $str);
				$reg = "/\t*<!--\s*BEGIN\s+$block\s+-->\n*(\s*.*?\n?)\t*<!--\s+END\s+$block\s*-->\n*((\s*.*?\n?)\t*<!--\s+FINALLY\s+$block\s*-->\n?)?/sm";
			}
			else $reg = "/<!--\s*BEGIN\s+$block\s+-->\s*(\s*.*?\s*)<!--\s+END\s+$block\s*-->\s*((\s*.*?\s*)<!--\s+FINALLY\s+$block\s*-->)?\s*/sm";
			if(1!==preg_match($reg, $str, $m)) throw new \UnexpectedValueException("mal-formed block $block");
			$this->setValue($name, '');
			$this->setValue($block, $m[1]);
			$this->setValue($parent, preg_replace($reg, "{".$name."}", $str));
			if(isset($m[3])) $this->finally[$block] = $m[3];
		}

		protected function setValue($varname, $value) {
			$this->values['{'.$varname.'}'] = $value;
		}

		private function getVar($varname) {
			return $this->values['{'.$varname.'}'];
		}

		public function clear($varname) {
			$this->setValue($varname, "");
		}

		public function setParent($parent, $block){
			$this->parents[$parent][] = $block;
		}

		private function substModifiers($value, $exp){
			$statements = explode('|', $exp);
			for($i=1; $i<sizeof($statements); $i++){
				$temp = explode(":", $statements[$i]);
				$function = $temp[0];
				$parameters = array_diff($temp, array($function));
				$value = call_user_func_array($function, array_merge(array($value), $parameters));
			}
			return $value;
		}

		protected function subst($value) {
			// Common variables replacement
			$s = str_replace(array_keys($this->values), $this->values, $value);
			// Common variables with modifiers
			foreach($this->modifiers as $var => $expressions){
				if(false!==strpos($s, "{".$var."|")) foreach($expressions as $exp){
					if(false===strpos($var, "->") && isset($this->values['{'.$var.'}'])){
						$s = str_replace('{'.$exp.'}', $this->substModifiers($this->values['{'.$var.'}'], $exp), $s);
					}
				}
			}
			// Object variables replacement
			foreach($this->instances as $var => $instance){
				foreach($this->properties[$var] as $properties){
					if(false!==strpos($s, "{".$var.$properties."}") || false!==strpos($s, "{".$var.$properties."|")){
						$pointer = $instance;
						$property = explode("->", $properties);
						for($i = 1; $i < sizeof($property); $i++){
							if(!is_null($pointer)){
								$obj = strtolower(str_replace('_', '', $property[$i]));
								// Get accessor
								if(method_exists($pointer, "get$obj")) $pointer = $pointer->{"get$obj"}();
								// Magic __get accessor
								elseif(method_exists($pointer, "__get")) $pointer = $pointer->__get($property[$i]);
								// Property acessor
								elseif(property_exists($pointer, $obj)) $pointer = $pointer->$obj;
								else {
									$className = $property[$i-1] ? $property[$i-1] : get_class($instance);
									$class = is_null($pointer) ? "NULL" : get_class($pointer);
									throw new \BadMethodCallException("no accessor method in class ".$class." for ".$className."->".$property[$i]);
								}
							} else {
								$pointer = $instance->get($obj);
							}
						}
						// Checking if final value is an object...
						if(is_object($pointer)){
							$pointer = method_exists($pointer, "__toString") ? $pointer->__toString() : "Object";
						// ... or an array
						} elseif(is_array($pointer)){
							$value = "";
							for($i=0; list($key, $val) = each($pointer); $i++){
								$value.= "$key => $val";
								if($i<sizeof($pointer)-1) $value.= ",";
							}
							$pointer = $value;
						}
						// Replacing value
						$s = str_replace("{".$var.$properties."}", $pointer, $s);
						// Object with modifiers
						if(isset($this->modifiers[$var.$properties])){
							foreach($this->modifiers[$var.$properties] as $exp){
								$s = str_replace('{'.$exp.'}', $this->substModifiers($pointer, $exp), $s);
							}
						}
					}
				}
			}
			return $s;
		}

		public function block($block) {
			#if(!in_array($block, $this->blocks)) throw new \InvalidArgumentException("block $block does not exist");
			// Checking finally blocks inside this block
			if(isset($this->parents[$block])) foreach($this->parents[$block] as $child){
				if(isset($this->finally[$child]) && !in_array($child, $this->parsed)){
					$this->setValue($child.'_value', $this->subst($this->finally[$child]));
					$this->parsed[] = $block;
				}
			}
			$this->setValue($block.'_value', $this->getVar($block.'_value') . $this->subst($this->getVar($block)));
			if(!in_array($block, $this->parsed)) $this->parsed[] = $block;
			// Cleaning children
			if(isset($this->parents[$block])) foreach($this->parents[$block] as $child) $this->clear($child.'_value');
		}

		public function parse() {
			// Auto assistance for parse children blocks
			foreach(array_reverse($this->parents) as $parent => $children){
				foreach($children as $block){
					if(in_array($parent, $this->blocks) && in_array($block, $this->parsed) && !in_array($parent, $this->parsed)){
						$this->setValue($parent.'_value', $this->subst($this->getVar($parent)));
						$this->parsed[] = $parent;
					}
				}
			}
			// Parsing finally blocks
			foreach($this->finally as $block => $content){
				if(!in_array($block, $this->parsed)){
					$this->setValue($block.'_value', $this->subst($content));
				}
			}
			// After subst, remove empty vars
			return preg_replace("/{(".self::$REG_NAME.")((\-\>(".self::$REG_NAME."))*)?((\|.*?)*)?}/", "", $this->subst($this->getVar(".")));
		}

		public function show() {
			echo $this->parse();
		}

	}
